<?php
use Codexpert\CoDesigner_Pro\Helper;
use Mpdf\Mpdf;

/**
 * Generate pdf invoice
 * 
 * @param string $order_id 
 * @param string $invoice_content 
 * 
 * @return string $invoice_file_path
 */
if( ! function_exists( 'codesigner_generate_pdf_invoice' ) ) :
function codesigner_generate_pdf_invoice( $order_id, $invoice_content ) {
    // create invoice file path
    $invoice_file_path   = codesigner_get_invoice_path( $order_id );

    // validate file path if already exists
    if ( file_exists( $invoice_file_path ) ) return $invoice_file_path;

    ob_start();

    // Initialize mPDF
    $mpdf = new Mpdf();

    // Write HTML content
    $mpdf->WriteHTML( $invoice_content );

    // Save the PDF
    $mpdf->Output( $invoice_file_path, 'F' );

    ob_get_clean();

    return $invoice_file_path;
}
endif;

/**
 * Get invoice file path
 * 
 * @param string $order_id 
 * 
 * @return string $invoice_file_path
 */
if( ! function_exists( 'codesigner_get_invoice_path' ) ) :
function codesigner_get_invoice_path( $order_id ) {
    // create invoice file path
    $invoice_dir            = codesigner_get_invoice_dir();
    $invoice_file_path      = $invoice_dir . '/invoice-' . $order_id . '.pdf';

    return $invoice_file_path;
}
endif;

/**
 * Get codesigner invoice directory path
 * 
 * @return string codesigner invoice directory path
 */
if( ! function_exists( 'codesigner_get_invoice_dir' ) ) :
function codesigner_get_invoice_dir() {
    return trailingslashit( wp_upload_dir()['basedir'] ) . 'codesigner-invoices';
}
endif;

/**
 * Get invoice download URL
 * 
 * @param string $order_id 
 * 
 * @return string $invoice_url
 */
if( ! function_exists( 'codesigner_get_invoice_download_url' ) ) :
function codesigner_get_invoice_download_url( $order_id ) {
    $upload_dir     = wp_upload_dir();
    $baseurl 	    = $upload_dir['baseurl'];
    $invoice_url    = sprintf( '%1s/codesigner-invoices/invoice-%2s.pdf', $baseurl, $order_id );

    return $invoice_url;
}
endif;

/**
 * Get elementor invoice template css
 * 
 * @param string $template_id 
 * 
 * @return string $css
 */
if( ! function_exists( 'codesigner_get_invoice_template_css' ) ) :
function codesigner_get_invoice_template_css( $template_id ) {
    $css_files   = [];
    $css_files[] = trailingslashit( ELEMENTOR_ASSETS_URL ) . 'css/frontend.min.css';
    $css_files[] = trailingslashit( ELEMENTOR_ASSETS_URL ) . 'css/common.min.css';
    $css_files[] = trailingslashit( ELEMENTOR_ASSETS_URL ) . 'lib/font-awesome/css/font-awesome.min.css';
    $css_files[] = trailingslashit( ELEMENTOR_ASSETS_URL ) . 'lib/eicons/css/elementor-icons.min.css';
    $css_files[] = trailingslashit( ELEMENTOR_ASSETS_URL ) . 'lib/font-awesome/css/solid.min.css';
    $css_files[] = trailingslashit( ELEMENTOR_ASSETS_URL ) . 'lib/font-awesome/css/brands.min.css';
    $css_files[] = trailingslashit( CODESIGNER_ASSETS ) . 'css/front.css';
    $css_files[] = trailingslashit( CODESIGNER_ASSETS ) . 'css/cx-grid.css';

    global $wp_styles;
    foreach( $wp_styles->queue as $_style ) :
        $css_files[] = $wp_styles->registered[ $_style ]->src;
    endforeach;

    $css = '';
    foreach ( $css_files as $key=>$css_file ) {
        if( strpos( $css_file, 'admin-bar.min.css' ) == false ) {
            $css .= file_get_contents( $css_file );
        }
    }

    // some common error fix
    $css .= ".elementor-widget-container{width:100%;}table {border-spacing: 0;border-collapse: collapse;width:100%}table td{width:200px;padding:10px; border: 1px solid #000}";

    // rewrite post css stuff
    $upload_dir = wp_get_upload_dir();
    $post_css   = file_get_contents( "{$upload_dir['basedir']}/elementor/css/post-{$template_id}.css" );
    $post_css   = preg_replace( '/@media\(min-width:\d+px\)\{/', '', $post_css );
    $post_css   = str_replace( '}}', '}', $post_css );
    $css        .= $post_css;
    $css        = str_replace( '--', '', $css );

    // add float left
    $pattern        = '/width\s*:\s*(\d+(\.\d+)?)%/i';
    $replacement    = function( $matches ) {
        $widthValue = $matches[1]; // Get the matched width value
        return "float:left;width:$widthValue%!important;";
    };

    return $css;
}
endif;

/**
 * Get invoice default css
 * 
 * @return string css
 */
if( ! function_exists( 'codesigner_get_default_invoice_css' ) ):
function codesigner_get_default_invoice_css() {
    $url        = dirname( CODESIGNER_PRO ) . "/modules/invoice-builder/assets/css/invoice.css";
    $context    = array(
        "ssl"   => array(
            "verify_peer"       => false,
            "verify_peer_name"  => false,
        ),
    );

    return file_get_contents( $url, false, stream_context_create( $context ) );
}
endif;

/**
 * Generate invoice html for pdf
 * 
 * @param string $default_styles 
 * @param string $invoice_styles 
 * @param string $invoice_content 
 * 
 * @return string $invoice_pdf_html
 */
if( ! function_exists( 'codesigner_get_invoice_html' ) ) :
function codesigner_get_invoice_template_html( $default_styles, $invoice_styles, $invoice_content ) {
    $invoice_pdf_html = sprintf( 
        '<html lang="en">
            <head>
            <meta charset="UTF-8" />
            <meta name="viewport" content="width=device-width, initial-scale=1.0" />
            <title>%1s</title>
            <style>%2s</style>
            <style>%3s</style>
            </head>
            <body>%3s</body>
        </html>', 
        __( 'Invoice', 'codesigner-pro' ), 
        $default_styles,
        $invoice_styles, 
        $invoice_content, 
    );

    return $invoice_pdf_html;
}
endif;

/**
 * Remove styles inside style tag form content 
 * 
 * @param string $string 
 * 
 * @return string $string
 */
if( ! function_exists( 'codesigner_invoice_content_remove_styles' ) ) :
function codesigner_invoice_content_remove_styles( $string ) {
    $startPos   = strpos( $string, '<style>' );
    $endPos     = strpos( $string, '</style>', $startPos );

    if ( $startPos !== false && $endPos !== false ) {
        $string = substr( $string, 0, $startPos ) . substr( $string, $endPos + strlen( '</style>' ) );
    }

    return $string;
}
endif;

/**
 * Load Elementor invoice template inside widget
 * 
 * @param array $args 
 * 
 * @return 
 */
if( ! function_exists( 'codesigner_get_invoice_template' ) ) :
function codesigner_get_invoice_template( $args ) {
    // default template directory
    $invoice_template_dir   = dirname( CODESIGNER_PRO ) . "/modules/invoice-builder/templates/";

    // full path of a template file in module directory
    $invoice_template_path  = $invoice_template_dir . $args['template-slug'] . '.php';

    if ( file_exists( $invoice_template_path ) ) {
        ob_start();
        include $invoice_template_path;
        return ob_get_clean();
    }
    else return;
}
endif;
